// app/api/downloads/route.ts
import { NextResponse } from "next/server"
import type { NextRequest } from "next/server"
import { getAccessItem } from "@/lib/access/catalog"
import { adminDb } from "@/lib/firebaseAdmin"
import { requireFirebaseUser } from "@/lib/auth/requireIdToken"
import type { DownloadApiResponse } from "@/types/access"

console.log("CLIENT projectId:", process.env.NEXT_PUBLIC_FIREBASE_PROJECT_ID);
console.log("ADMIN projectId:", JSON.parse(process.env.FIREBASE_SERVICE_ACCOUNT_KEY || "{}")?.project_id);


export async function GET(req: NextRequest) {
  const { searchParams } = new URL(req.url)
  const sku = searchParams.get("sku")

  const item = getAccessItem(sku)
  if (!item) {
    const res: DownloadApiResponse = { ok: false, error: "Invalid sku" }
    return NextResponse.json(res, { status: 400 })
  }

  // 1) Verify signed-in user via Firebase ID token
  const user = await requireFirebaseUser(req)
  if (!user.ok) {
    const res: DownloadApiResponse = { ok: false, error: user.error }
    return NextResponse.json(res, { status: 401 })
  }

  // 2) Check payment flag in Firestore
  const db = adminDb()
  const userRef = db.collection("users").doc(user.uid)
  const snap = await userRef.get()

  const data = snap.exists ? snap.data() : null
  const hasFullAccess = Boolean(data?.hasFullAccess)

  if (!hasFullAccess) {
    const res: DownloadApiResponse = { ok: false, error: "Payment required" }
    return NextResponse.json(res, { status: 402 })
  }

  // 3) Return the URL (public for MVP, still gated by this API)
  const res: DownloadApiResponse = {
    ok: true,
    sku: item.sku,
    title: item.title,
    url: item.publicPath,
  }

  return NextResponse.json(res, { status: 200 })
}
